#!/usr/bin/env bash
# nostromo.sh — Alien-themed terminal navigation game
# Author: ChatGPT (for Chelsey)
#
# Usage:
#   bash nostromo.sh init   # create the game world in ./USCSS_Nostromo
#   bash nostromo.sh reset  # delete and recreate the world
#   bash nostromo.sh check  # check if you completed the objective
#
# Game premise:
#   You are aboard the USCSS Nostromo. Xenomorphs have nested somewhere in the
#   ship. Use only basic terminal commands (pwd, ls/-a, cd, cat, less, mv,
#   rm -r) to explore directories, read logs, move items, and ultimately delete
#   the directory where the xenomorphs are hiding. Do NOT delete the whole
#   ship. We still need the security deposit back.
#
# Objective:
#   Find and `rm -r` the *xenomorph_nest* directory hidden somewhere on the
#   ship.
#   Optional bonus: stabilize the reactor by moving the coolant_rod into place.
#
# Notes:
#   - Everything is plain files and folders. No scripts need to be executed to
#     play.
#   - Hints and flavor text live in .log or .txt files. Some are hidden. Use
#     `ls -a`.
#   - You never need sudo. Please don’t play this as root.

set -euo pipefail

WORLD_DIR="USCSS_Nostromo"
TARGET_DIR="$WORLD_DIR/maintenance/.ducts/gamma/xenomorph_nest"
DECOY1_DIR="$WORLD_DIR/cargo/bay2/storage"
DECOY2_DIR="$WORLD_DIR/engineering/walkway/crew_storage"
COOLANT_SOURCE="$WORLD_DIR/cargo/bay1/coolant_rod.txt"
COOLANT_TARGET_DIR="$WORLD_DIR/engineering/reactor"

say() {
  printf "%s
" "$*"
}

write_file() {
  # $1 = path, $2... = content
  local path="$1"
  shift || true
  mkdir -p "$(dirname "$path")"
  cat >"$path" <<'EOF'
EOF
  printf "%s
" "$@" >>"$path"
}

populate_world() {
  say "Creating USCSS Nostromo…"

  # Top-level sections
  mkdir -p \
    "$WORLD_DIR/bridge" \
    "$WORLD_DIR/medbay/scans" \
    "$WORLD_DIR/engineering/reactor" \
    "$WORLD_DIR/engineering/walkway" \
    "$WORLD_DIR/maintenance/locker" \
    "$WORLD_DIR/cargo/bay1" \
    "$WORLD_DIR/cargo/bay2"

  # Hidden ducts with branches
  mkdir -p \
    "$WORLD_DIR/maintenance/.ducts/alpha" \
    "$WORLD_DIR/maintenance/.ducts/beta" \
    "$WORLD_DIR/maintenance/.ducts/gamma"

  # Real nest location (now in gamma)
  mkdir -p "$TARGET_DIR/eggs" "$TARGET_DIR/chitin"

  # Decoys to test restraint
  mkdir -p "$DECOY1_DIR/empty_crates" "$DECOY2_DIR/tools"

  # Flavor + hints
  write_file "$WORLD_DIR/README.md" \
"# USCSS Nostromo — Shipboard Terminal Training" "" \
"Use \`pwd\`, \`ls\` (and \`ls -a\`), \`cd\`, \`cat\`, \`less\`, \`mv\`," \
"and \`rm -r\` to explore and complete your mission." "" \
"**Primary Objective:** Locate and delete the directory named" \
"*xenomorph_nest*. It might be hidden!" "" \
"**Caution:** Do not delete the ship (\`USCSS_Nostromo\`). Collateral" \
"damage may void Weyland-Yutani warranties." "" \
"**Bonus Objective:** Move the coolant rod into the reactor" 

  write_file "$WORLD_DIR/bridge/crew_manifest.log" \
"USCSS Nostromo — Crew Manifest (Partial)" \
"Ripley, Dallas, Lambert, Kane, Parker, Brett, Ash" \
"Note: We cleared the cargo. No alien lifeform detected." \
"Lambert: 'If it's not on the map, check the ducts.'"

  write_file "$WORLD_DIR/medbay/scans/unknown_lifeform.txt" \
"— MedScanner v2.1 —" \
"Thermal hotspots near maintenance; readings bifurcate in three branches:" \
"alpha, beta, gamma." \
"Beta and alpha show nothing conclusive. Gamma spikes periodically."

  write_file "$WORLD_DIR/engineering/walkway/maintenance_note.txt" \
"Walkway cams down. Heard scraping behind bulkhead panels." \
"If you need reactor stability, slot the coolant rod in reactor bay." \
"(cargo/bay1 → engineering/reactor)"

  write_file "$WORLD_DIR/cargo/bay1/cargo_manifest.log" \
"Crates: mining equipment, rations, replacement ducts." \
"Loose item: coolant_rod.txt (handle with care)."

  write_file "$WORLD_DIR/cargo/bay1/coolant_rod.txt" \
"Coolant Rod — Insert in reactor to reduce core temperature." \
"Instructions: Move this file to engineering/reactor/"
  
  write_file "$WORLD_DIR/maintenance/locker/inventory.txt" \
"- Plasma Torch" \
"- Solder" \
"- Various hand tools"

  # Hidden hints in the ducts (gamma is hot now)
  write_file "$WORLD_DIR/maintenance/.ducts/beta/status.log" \
"Beta branch clear. Old molting husks, no active heat signatures."

  write_file "$WORLD_DIR/maintenance/.ducts/alpha/status.log" \
"Alpha branch: nothing but dust and a very judgmental space rat."

  write_file "$WORLD_DIR/maintenance/.ducts/gamma/tracks.log" \
"Fresh acid scarring on grates. Movement sounds toward deeper vents." \
"If you are reading this: step lightly."

  write_file "$TARGET_DIR/hiss.log" \
"…sssss…" "Footsteps echo. Something retreats deeper into the shadows."

  write_file "$TARGET_DIR/eggs/ovipositor.log" \
"Rows of leathery eggs line the chamber. Do not tap to interact."

  write_file "$TARGET_DIR/chitin/resin_notes.txt" \
"Walls cocooned. Structure looks… purposeful. This is definitely the nest."

  # Decoy flavor
  write_file "$DECOY1_DIR/label.txt" \
"These are definitely not eggs. Just crates. Super boring crates."

  write_file "$DECOY2_DIR/tools/inventory.txt" \
"Spanners, welders, half a donut. No monsters."

  # A tiny trap for overeager `rm -r *` enjoyers
  write_file "$WORLD_DIR/.insurance/CLAIMS_README.txt" \
"Hello! If you can read this, you used ls -a. Proud of you." \
"Reminder: Deleting the entire ship voids your bonus."

  say "World created at ./$WORLD_DIR"
  say "Open $WORLD_DIR/README.md to begin. Good hunting."
}

check_objective() {
  local status=0

  if [[ ! -d "$WORLD_DIR" ]]; then
    say "❌ Ship missing. Did you delete the whole Nostromo?"
    return 1
  fi

  if [[ -d "$TARGET_DIR" ]]; then
    say "❌ Xenomorph nest still present: $TARGET_DIR"
    status=1
  else
    say "✅ Primary objective complete: nest eliminated."
  fi

  if [[ ! -d "$DECOY1_DIR" || ! -d "$DECOY2_DIR" ]]; then
    say "⚠️  Collateral damage detected. You deleted more than necessary."
    status=1
  fi

  if [[ -f "$COOLANT_TARGET_DIR/$(basename "$COOLANT_SOURCE")" ]]; then
    say "🌡️  Bonus complete: Reactor stabilized."
  else
    if [[ -f "$COOLANT_SOURCE" ]]; then
      say "ℹ️  Bonus available: Move coolant_rod.txt to engineering/reactor/"
    else
      say "ℹ️  Coolant rod not found in cargo. If you moved it elsewhere, consider" \
          "placing it in the reactor."
    fi
  fi

  if [[ $status -eq 0 ]]; then
    say "🎉 Mission success. Weyland-Yutani will absolutely still underpay you."
  else
    say "Run \"bash nostromo.sh check\" again after fixing issues."
  fi

  return $status
}

reset_world() {
  if [[ -d "$WORLD_DIR" ]]; then
    say "Removing existing world…"
    rm -rf "$WORLD_DIR"
  fi
  populate_world
}

cmd=${1:-}
case "${cmd}" in
  init)
    if [[ -e "$WORLD_DIR" ]]; then
      say "Directory $WORLD_DIR already exists. Use 'reset' to recreate, or start" \
          "playing!"
    else
      populate_world
    fi
    ;;
  reset)
    reset_world
    ;;
  check)
    check_objective
    ;;
  *)
    cat <<USAGE
nostromo.sh — Alien terminal navigation game

Usage:
  bash nostromo.sh init   # create the game world in ./USCSS_Nostromo
  bash nostromo.sh reset  # delete and recreate the world
  bash nostromo.sh check  # check progress

Then play using only: pwd, ls (and -a), cd, cat, less, mv, rm -r
USAGE
    ;;
 esac
