#!/usr/bin/env bash
# env_justice.sh — Environmental Justice Project (terminal practice)
# Author: ChatGPT (for Chelsey)
#
# Usage:
#   bash env_justice.sh init   # create the world in ./EnvJustice
#   bash env_justice.sh reset  # delete and recreate the world
#   bash env_justice.sh check  # check the win conditions
#
# Premise:
#   Marginalized neighborhoods have been burdened by hidden pollutant
#   directories. Your job is to remove pollutants and help restore
#   community resources by organizing clean files into community_gardens/.
#
# Play using only: pwd, ls (and ls -a), cd, cat, less, mv, rm -r
#
# Objectives (win conditions):
#   1) Remove pollutant directories: .toxic_dump and industrial_waste
#   2) Move clean resources into community_gardens/
#   3) Keep core neighborhood folders intact
#
set -euo pipefail

WORLD_DIR="EnvJustice"
GARDEN_DIR="$WORLD_DIR/community_gardens"

POLLUTANTS=(
  "$WORLD_DIR/neighborhoods/alpha/.toxic_dump"
  "$WORLD_DIR/neighborhoods/beta/industrial_waste"
  "$WORLD_DIR/neighborhoods/gamma/.toxic_dump"
)

CLEAN_FILES=(
  "alpha_water_quality_clean.csv"
  "beta_tree_planting_plan.txt"
  "gamma_air_readings_clean.csv"
)

say() { printf "%s\n" "$*"; }

write_file() {
  local path="$1"
  shift || true
  mkdir -p "$(dirname "$path")"
  cat >"$path" <<'EOF'
EOF
  printf "%s\n" "$@" >>"$path"
}

populate_world() {
  say "Creating EnvJustice world…"

  mkdir -p \
    "$WORLD_DIR/neighborhoods/alpha" \
    "$WORLD_DIR/neighborhoods/beta" \
    "$WORLD_DIR/neighborhoods/gamma" \
    "$GARDEN_DIR" \
    "$WORLD_DIR/docs" \
    "$WORLD_DIR/reports"

  # Place pollutants (hidden dirs and junk)
  mkdir -p "$WORLD_DIR/neighborhoods/alpha/.toxic_dump"
  write_file \
    "$WORLD_DIR/neighborhoods/alpha/.toxic_dump/leak.log" \
    "Contaminant levels spike near old pipeline."

  mkdir -p "$WORLD_DIR/neighborhoods/beta/industrial_waste"
  write_file \
    "$WORLD_DIR/neighborhoods/beta/industrial_waste/readme.txt" \
    "Improperly stored sludge. Remove to protect residents."

  mkdir -p "$WORLD_DIR/neighborhoods/gamma/.toxic_dump"
  write_file \
    "$WORLD_DIR/neighborhoods/gamma/.toxic_dump/notes.log" \
    "Residents report strong odors and discoloration."

  # Clean resource files scattered in neighborhoods
  write_file \
    "$WORLD_DIR/neighborhoods/alpha/alpha_water_quality_clean.csv" \
    "site,ph,contaminant_ppm" \
    "A1,7.2,0.3"

  write_file \
    "$WORLD_DIR/neighborhoods/beta/beta_tree_planting_plan.txt" \
    "Plant native trees along main street to cool neighborhoods."

  write_file \
    "$WORLD_DIR/neighborhoods/gamma/gamma_air_readings_clean.csv" \
    "site,pm25,pm10" \
    "G1,12,20"

  # Docs and hints
  write_file "$WORLD_DIR/docs/README.md" \
    "EnvJustice — cleanup practice" \
    "Find and remove pollutant directories, then move clean" \
    "resources into community_gardens/. Use ls -a to find hidden" \
    "directories."

  write_file "$WORLD_DIR/reports/ethics_note.txt" \
    "Environmental justice work prioritizes impacted communities." \
    "Do not delete core neighborhood directories."

  say "World created at ./$WORLD_DIR"
  say "Open $WORLD_DIR/docs/README.md to begin."
}

check_objective() {
  local status=0

  if [[ ! -d "$WORLD_DIR" ]]; then
    say "❌ World missing. Did you delete EnvJustice?"
    return 1
  fi

  # Pollutants should be gone
  for p in "${POLLUTANTS[@]}"; do
    if [[ -d "$p" ]]; then
      say "❌ Pollutant still present: $p"
      status=1
    fi
  done

  # Clean files should be in community_gardens/
  for f in "${CLEAN_FILES[@]}"; do
    if [[ ! -f "$GARDEN_DIR/$f" ]]; then
      say "❌ Clean resource missing in gardens: $f"
      status=1
    fi
  done

  # Ensure core neighborhoods still exist
  for n in "$WORLD_DIR/neighborhoods/alpha" \
    "$WORLD_DIR/neighborhoods/beta" \
    "$WORLD_DIR/neighborhoods/gamma"; do
    if [[ ! -d "$n" ]]; then
      say "⚠️  Collateral damage: missing neighborhood $n"
      status=1
    fi
  done

  if [[ $status -eq 0 ]]; then
    say "✅ Mission success: pollutants removed and gardens stocked."
  else
    say "Run 'bash env_justice.sh check' after fixing issues."
  fi

  return $status
}

reset_world() {
  if [[ -d "$WORLD_DIR" ]]; then
    say "Removing existing world…"
    rm -rf "$WORLD_DIR"
  fi
  populate_world
}

cmd=${1:-}
case "$cmd" in
  init)
    if [[ -e "$WORLD_DIR" ]]; then
      say "Directory $WORLD_DIR already exists. Use 'reset' to recreate." \
          "Or start playing!"
    else
      populate_world
    fi
    ;;
  reset)
    reset_world
    ;;
  check)
    check_objective
    ;;
  *)
    cat <<USAGE
env_justice.sh — Environmental Justice Project (terminal practice)

Usage:
  bash env_justice.sh init   # create the world in ./EnvJustice
  bash env_justice.sh reset  # delete and recreate the world
  bash env_justice.sh check  # check the win conditions

Play using: pwd, ls (and -a), cd, cat, less, mv, rm -r
USAGE
    ;;
esac
